'use strict';

/* --------------------------------------------------------------
 init.js 2016-10-11
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Invoices Table Controller
 *
 * This controller initializes the main invoices table with a new jQuery DataTables instance.
 */
gx.controllers.module('init', [jse.source + '/vendor/datatables/jquery.dataTables.min.css', jse.source + '/vendor/datatables/jquery.dataTables.min.js', jse.source + '/vendor/momentjs/moment.min.js', gx.source + '/libs/invoices_overview_columns', gx.source + '/libs/search', 'datatable', 'modal', 'user_configuration_service'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Get Initial Table Order
     *
     * @param {Object} parameters Contains the URL parameters.
     * @param {Object} columns Contains the column definitions.
     *
     * @return {Array[]}
     */
    function _getOrder(parameters, columns) {
        var index = 1; // Order by first column by default.
        var direction = 'desc'; // Order DESC by default.

        // Apply initial table sort.
        if (parameters.sort) {
            direction = parameters.sort.charAt(0) === '-' ? 'desc' : 'asc';
            var columnName = parameters.sort.slice(1);

            var _iteratorNormalCompletion = true;
            var _didIteratorError = false;
            var _iteratorError = undefined;

            try {
                for (var _iterator = columns[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                    var column = _step.value;

                    if (column.name === columnName) {
                        index = columns.indexOf(column);
                        break;
                    }
                }
            } catch (err) {
                _didIteratorError = true;
                _iteratorError = err;
            } finally {
                try {
                    if (!_iteratorNormalCompletion && _iterator.return) {
                        _iterator.return();
                    }
                } finally {
                    if (_didIteratorError) {
                        throw _iteratorError;
                    }
                }
            }
        } else if (data.activeColumns.indexOf('invoiceDate') > -1) {
            // Order by date if possible.
            index = data.activeColumns.indexOf('invoiceDate');
        }

        return [[index, direction]];
    }

    /**
     * Get Initial Search Cols
     *
     * @param {Object} parameters Contains the URL parameters.
     *
     * @returns {Object[]} Returns the initial filtering values.
     */
    function _getSearchCols(parameters, columns) {
        if (!parameters.filter) {
            return [];
        }

        var searchCols = [];

        var _iteratorNormalCompletion2 = true;
        var _didIteratorError2 = false;
        var _iteratorError2 = undefined;

        try {
            for (var _iterator2 = columns[Symbol.iterator](), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
                var column = _step2.value;

                var entry = null;
                var value = parameters.filter[column.name];

                if (value) {
                    entry = { search: decodeURIComponent(value) };
                }

                searchCols.push(entry);
            }
        } catch (err) {
            _didIteratorError2 = true;
            _iteratorError2 = err;
        } finally {
            try {
                if (!_iteratorNormalCompletion2 && _iterator2.return) {
                    _iterator2.return();
                }
            } finally {
                if (_didIteratorError2) {
                    throw _iteratorError2;
                }
            }
        }

        return searchCols;
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        var columns = jse.libs.datatable.prepareColumns($this, jse.libs.invoices_overview_columns, data.activeColumns);
        var parameters = $.deparam(window.location.search.slice(1));
        var pageLength = parseInt(parameters.length || data.pageLength);

        jse.libs.datatable.create($this, {
            autoWidth: false,
            dom: 't',
            pageLength: pageLength,
            displayStart: parseInt(parameters.page) ? (parseInt(parameters.page) - 1) * pageLength : 0,
            serverSide: true,
            language: jse.libs.datatable.getTranslations(jse.core.config.get('languageCode')),
            ajax: {
                url: jse.core.config.get('appUrl') + '/admin/admin.php?do=InvoicesOverviewAjax/DataTable',
                type: 'POST',
                data: {
                    pageToken: jse.core.config.get('pageToken')
                }
            },
            orderCellsTop: true,
            order: _getOrder(parameters, columns),
            searchCols: _getSearchCols(parameters, columns),
            columns: columns
        });

        // Add table error handler.
        jse.libs.datatable.error($this, function (event, settings, techNote, message) {
            var title = 'DataTables ' + jse.core.lang.translate('error', 'messages');
            jse.libs.modal.showMessage(title, message);
        });

        // Add pagination change handler.
        $this.on('datatable_custom_pagination:length_change', function (event, newPageLength) {
            jse.libs.user_configuration_service.set({
                data: {
                    userId: jse.core.registry.get('userId'),
                    configurationKey: 'invoicesOverviewPageLength',
                    configurationValue: newPageLength
                }
            });
        });

        // Add draw event handler.
        $this.on('draw.dt', function () {
            $this.find('thead input:checkbox').prop('checked', false).trigger('change', [false]); // No need to update the tbody checkboxes (event.js).
            $this.find('tbody').attr('data-gx-widget', 'single_checkbox');
            gx.widgets.init($this); // Initialize the checkbox widget.
        });

        // Set admin search value.
        if (Object.keys(parameters).includes('filter') && Object.keys(parameters.filter).includes('invoiceNumber')) {
            jse.libs.search.setValue(parameters.filter.invoiceNumber, true);
        }

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
